# BSD 3-Clause License; see https://github.com/scikit-hep/uproot5/blob/main/LICENSE

"""
This module defines behaviors for :doc:`uproot.behaviors.RNTuple.RNTuple` and
:doc:`uproot.behaviors.RNTuple.HasFields` (both ``RField`` and
``RNTuple``).

Most of the functionality of RNTuple-reading is implemented here.

See :doc:`uproot.models.RNTuple` for deserialization of the ``RNTuple``
objects themselves.
"""
from __future__ import annotations

import sys
import warnings
from collections.abc import Mapping
from functools import partial

import numpy

import uproot
import uproot.interpretation.grouped
import uproot.language.python
import uproot.source.chunk
from uproot._util import no_filter, unset
from uproot.behaviors.TBranch import _regularize_array_cache


def iterate(
    files,
    expressions=None,  # TODO: Not implemented yet
    cut=None,  # TODO: Not implemented yet
    *,
    filter_name=no_filter,
    filter_typename=no_filter,
    filter_field=no_filter,
    aliases=None,  # TODO: Not implemented yet
    language=uproot.language.python.python_language,  # TODO: Not implemented yet
    step_size="100 MB",
    decompression_executor=None,  # TODO: Not implemented yet
    library="ak",  # TODO: Not implemented yet
    ak_add_doc=False,
    how=None,
    report=False,  # TODO: Not implemented yet
    allow_missing=False,  # TODO: Not implemented yet
    # For compatibility reasons we also accepts kwargs meant for TTrees
    filter_branch=unset,
    interpretation_executor=unset,
    custom_classes=unset,
    **options,
):
    """
    Args:
        files: See below.
        expressions (None, str, or list of str): Names of ``RFields`` or
            aliases to convert to arrays or mathematical expressions of them.
            Uses the ``language`` to evaluate. If None, all ``RFields``
            selected by the filters are included. (Not implemented yet.)
        cut (None or str): If not None, this expression filters all of the
            ``expressions``. (Not implemented yet.)
        filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
            filter to select ``TBranches`` by name.
        filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
            filter to select ``TBranches`` by type.
        filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool, or None): A
            filter to select ``RFields`` using the full
            :doc:`uproot.models.RNTuple.RField` object. If the function
            returns False or None, the ``RField`` is excluded; if the function
            returns True, it is included.
        aliases (None or dict of str \u2192 str): Mathematical expressions that
            can be used in ``expressions`` or other aliases.
            Uses the ``language`` engine to evaluate. (Not implemented yet.)
        language (:doc:`uproot.language.Language`): Language used to interpret
            the ``expressions`` and ``aliases``. (Not implemented yet.)
        step_size (int or str): If an integer, the maximum number of entries to
            include in each iteration step; if a string, the maximum memory size
            to include. The string must be a number followed by a memory unit,
            such as "100 MB".
        decompression_executor (None or Executor with a ``submit`` method): The
            executor that is used to decompress ``RPages``; if None, a
            :doc:`uproot.source.futures.TrivialExecutor` is created. (Not implemented yet.)
        library (str or :doc:`uproot.interpretation.library.Library`): The library
            that is used to represent arrays. Options are ``"np"`` for NumPy,
            ``"ak"`` for Awkward Array, and ``"pd"`` for Pandas. (Not implemented yet.)
        ak_add_doc (bool | dict ): If True and ``library="ak"``, add the RField ``description``
            to the Awkward ``__doc__`` parameter of the array.
            if dict = {key:value} and ``library="ak"``, add the RField ``value`` to the
            Awkward ``key`` parameter of the array.
        how (None, str, or container type): Library-dependent instructions
            for grouping. The only recognized container types are ``tuple``,
            ``list``, and ``dict``. Note that the container *type itself*
            must be passed as ``how``, not an instance of that type (i.e.
            ``how=tuple``, not ``how=()``).
        report (bool): If True, this generator yields
            (arrays, :doc:`uproot.behaviors.TBranch.Report`) pairs; if False,
            it only yields arrays. The report has data about the ``TFile``,
            ``TTree``, and global and local entry ranges.
        allow_missing (bool): If True, skip over any files that do not contain
            the specified ``RNTuple``.
        filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        interpretation_executor (None): This argument is not used and is only included for now
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        custom_classes (None): This argument is not used and is only included for now
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        options: See below.

    Iterates through contiguous chunks of entries from a set of files.

    For example:

    .. code-block:: python

        >>> for array in uproot.iterate("files*.root:ntuple", filter_names=["x", "y"], step_size=100):
        ...     # each of the following have 100 entries
        ...     array["x"], array["y"]

    Allowed types for the ``files`` parameter:

    * str/bytes: relative or absolute filesystem path or URL, without any colons
      other than Windows drive letter or URL schema.
      Examples: ``"rel/file.root"``, ``"C:\\abs\\file.root"``, ``"http://where/what.root"``
    * str/bytes: same with an object-within-ROOT path, separated by a colon.
      Example: ``"rel/file.root:tdirectory/rntuple"``
    * pathlib.Path: always interpreted as a filesystem path or URL only (no
      object-within-ROOT path), regardless of whether there are any colons.
      Examples: ``Path("rel:/file.root")``, ``Path("/abs/path:stuff.root")``
    * glob syntax in str/bytes and pathlib.Path.
      Examples: ``Path("rel/*.root")``, ``"/abs/*.root:tdirectory/rntuple"``
    * dict: keys are filesystem paths, values are objects-within-ROOT paths.
      Example: ``{{"/data_v1/*.root": "rntuple_v1", "/data_v2/*.root": "rntuple_v2"}}``
    * already-open RNTuple objects.
    * iterables of the above.

    Options (type; default): (Not implemented yet.)

    * handler (:doc:`uproot.source.chunk.Source` class; None)
    * timeout (float for HTTP, int for XRootD; 30)
    * max_num_elements (None or int; None)
        The maximum number of elements to be requested in a single vector read, when using XRootD.
    * num_workers (int; 1)
    * use_threads (bool; False on the emscripten platform (i.e. in a web browser), else True)
    * num_fallback_workers (int; 10)
    * begin_chunk_size (memory_size; 403, the smallest a ROOT file can be)
    * minimal_ttree_metadata (bool; True)

    See also :ref:`uproot.behaviors.RNTuple.HasFields.iterate` to iterate
    within a single file.

    Other file entry points:

    * :doc:`uproot.reading.open`: opens one file to read any of its objects.
    * :doc:`uproot.behaviors.RNTuple.iterate` (this function): iterates through
      chunks of contiguous entries in ``RNTuples``.
    * :doc:`uproot.behaviors.RNTuple.concatenate`: returns a single concatenated
      array from ``RNTuples``.
    * :doc:`uproot._dask.dask`: returns an unevaluated Dask array from ``RNTuples``.
    """
    files = uproot._util.regularize_files(files, steps_allowed=False, **options)
    library = uproot.interpretation.library._regularize_library(library)

    for file_path, object_path in files:
        hasfields = uproot._util.regularize_object_path(
            file_path, object_path, None, allow_missing, options
        )

        if hasfields is not None:
            with hasfields:
                try:
                    yield from hasfields.iterate(
                        expressions=expressions,
                        cut=cut,
                        filter_name=filter_name,
                        filter_typename=filter_typename,
                        filter_field=filter_field,
                        aliases=aliases,
                        language=language,
                        step_size=step_size,
                        decompression_executor=decompression_executor,
                        library=library,
                        ak_add_doc=ak_add_doc,
                        how=how,
                        report=report,
                        filter_branch=filter_branch,
                        interpretation_executor=interpretation_executor,
                    )

                except uproot.exceptions.KeyInFileError:
                    if allow_missing:
                        continue
                    else:
                        raise


def concatenate(
    files,
    expressions=None,  # TODO: Not implemented yet
    cut=None,  # TODO: Not implemented yet
    *,
    filter_name=no_filter,
    filter_typename=no_filter,
    filter_field=no_filter,
    aliases=None,  # TODO: Not implemented yet
    language=uproot.language.python.python_language,  # TODO: Not implemented yet
    entry_start=None,
    entry_stop=None,
    decompression_executor=None,  # TODO: Not implemented yet
    library="ak",  # TODO: Not implemented yet
    backend="cpu",
    interpreter="cpu",
    ak_add_doc=False,
    how=None,
    allow_missing=False,
    # For compatibility reasons we also accepts kwargs meant for TTrees
    filter_branch=unset,
    interpretation_executor=unset,
    custom_classes=unset,
    **options,
):
    """
    Args:
        files: See below.
        expressions (None, str, or list of str): Names of ``RFields`` or
            aliases to convert to arrays or mathematical expressions of them.
            Uses the ``language`` to evaluate. If None, all ``RFields``
            selected by the filters are included. (Not implemented yet.)
        cut (None or str): If not None, this expression filters all of the
            ``expressions``. (Not implemented yet.)
        filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
            filter to select ``TBranches`` by name.
        filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
            filter to select ``TBranches`` by type.
        filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool, or None): A
            filter to select ``RFields`` using the full
            :doc:`uproot.models.RNTuple.RField` object. If the function
            returns False or None, the ``RField`` is excluded; if the function
            returns True, it is included.
        aliases (None or dict of str \u2192 str): Mathematical expressions that
            can be used in ``expressions`` or other aliases.
            Uses the ``language`` engine to evaluate. (Not implemented yet.)
        language (:doc:`uproot.language.Language`): Language used to interpret
            the ``expressions`` and ``aliases``. (Not implemented yet.)
        entry_start (None or int): The first entry to include. If None, start
            at zero. If negative, count from the end, like a Python slice.
        entry_stop (None or int): The first entry to exclude (i.e. one greater
            than the last entry to include). If None, stop at
            :ref:`uproot.behaviors.RNTuple.HasFields.num_entries`. If negative,
            count from the end, like a Python slice.
        decompression_executor (None or Executor with a ``submit`` method): The
            executor that is used to decompress ``RPages``; if None, a
            :doc:`uproot.source.futures.TrivialExecutor` is created. (Not implemented yet.)
        library (str or :doc:`uproot.interpretation.library.Library`): The library
            that is used to represent arrays. Options are ``"np"`` for NumPy,
            ``"ak"`` for Awkward Array, and ``"pd"`` for Pandas. (Not implemented yet.)
        ak_add_doc (bool | dict ): If True and ``library="ak"``, add the RField ``description``
            to the Awkward ``__doc__`` parameter of the array.
            if dict = {key:value} and ``library="ak"``, add the RField ``value`` to the
            Awkward ``key`` parameter of the array.
        how (None, str, or container type): Library-dependent instructions
            for grouping. The only recognized container types are ``tuple``,
            ``list``, and ``dict``. Note that the container *type itself*
            must be passed as ``how``, not an instance of that type (i.e.
            ``how=tuple``, not ``how=()``).
        report (bool): If True, this generator yields
            (arrays, :doc:`uproot.behaviors.TBranch.Report`) pairs; if False,
            it only yields arrays. The report has data about the ``TFile``,
            ``TTree``, and global and local entry ranges.
        allow_missing (bool): If True, skip over any files that do not contain
            the specified ``RNTuple``.
        filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        interpretation_executor (None): This argument is not used and is only included for now
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        custom_classes (None): This argument is not used and is only included for now
            for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
            and will be removed in a future version.
        options: See below.

    Returns an array with data from a set of files concatenated into one.

    For example:

    .. code-block:: python

        >>> array = uproot.concatenate("files*.root:ntuple", filter_field=["x", "y"])

    Depending on the number of files, the number of selected ``RFields``, and
    the size of your computer's memory, this function might not have enough
    memory to run.

    Allowed types for the ``files`` parameter:

    * str/bytes: relative or absolute filesystem path or URL, without any colons
      other than Windows drive letter or URL schema.
      Examples: ``"rel/file.root"``, ``"C:\\abs\\file.root"``, ``"http://where/what.root"``
    * str/bytes: same with an object-within-ROOT path, separated by a colon.
      Example: ``"rel/file.root:tdirectory/rntuple"``
    * pathlib.Path: always interpreted as a filesystem path or URL only (no
      object-within-ROOT path), regardless of whether there are any colons.
      Examples: ``Path("rel:/file.root")``, ``Path("/abs/path:stuff.root")``
    * glob syntax in str/bytes and pathlib.Path.
      Examples: ``Path("rel/*.root")``, ``"/abs/*.root:tdirectory/rntuple"``
    * dict: keys are filesystem paths, values are objects-within-ROOT paths.
      Example: ``{{"/data_v1/*.root": "rntuple_v1", "/data_v2/*.root": "rntuple_v2"}}``
    * already-open RNTuple objects.
    * iterables of the above.

    Options (type; default): (Not implemented yet.)

    * handler (:doc:`uproot.source.chunk.Source` class; None)
    * timeout (float for HTTP, int for XRootD; 30)
    * max_num_elements (None or int; None)
        The maximum number of elements to be requested in a single vector read, when using XRootD.
    * num_workers (int; 1)
    * use_threads (bool; False on the emscripten platform (i.e. in a web browser), else True)
    * num_fallback_workers (int; 10)
    * begin_chunk_size (memory_size; 403, the smallest a ROOT file can be)
    * minimal_ttree_metadata (bool; True)

    Other file entry points:

    * :doc:`uproot.reading.open`: opens one file to read any of its objects.
    * :doc:`uproot.behaviors.RNTuple.iterate`: iterates through chunks of
      contiguous entries in ``RNTuples``.
    * :doc:`uproot.behaviors.RNTuple.concatenate` (this function): returns a
      single concatenated array from ``RNTuples``.
    * :doc:`uproot._dask.dask`: returns an unevaluated Dask array from ``RNTuples``.
    """
    files = uproot._util.regularize_files(files, steps_allowed=False, **options)
    library = uproot.interpretation.library._regularize_library(library)

    all_arrays = []
    global_start = 0
    global_stop = 0

    all_hasfields = []
    for file_path, object_path in files:
        _hasfields = uproot._util.regularize_object_path(
            file_path, object_path, None, allow_missing, options
        )
        if _hasfields is not None:
            all_hasfields.append(_hasfields)

    total_num_entries = sum(hasfields.num_entries for hasfields in all_hasfields)
    entry_start, entry_stop = uproot.behaviors.TBranch._regularize_entries_start_stop(
        total_num_entries, entry_start, entry_stop
    )
    for hasfields in all_hasfields:
        with hasfields:
            nentries = hasfields.num_entries
            global_stop += nentries

            if (
                global_start <= entry_start < global_stop
                or global_start < entry_stop <= global_stop
            ):
                # overlap, read only the overlapping entries
                local_entry_start = max(
                    0, entry_start - global_start
                )  # need to clip to 0
                local_entry_stop = entry_stop - global_start  # overflows are fine
            elif entry_start >= global_stop or entry_stop <= global_start:  # no overlap
                # outside of this file's range -> skip
                global_start = global_stop
                continue
            else:
                # read all entries
                local_entry_start = 0
                local_entry_stop = nentries

            try:
                arrays = hasfields.arrays(
                    expressions=expressions,
                    cut=cut,
                    filter_name=filter_name,
                    filter_typename=filter_typename,
                    filter_field=filter_field,
                    aliases=aliases,
                    language=language,
                    entry_start=local_entry_start,
                    entry_stop=local_entry_stop,
                    decompression_executor=decompression_executor,
                    array_cache=None,
                    library=library,
                    backend=backend,
                    interpreter=interpreter,
                    ak_add_doc=ak_add_doc,
                    how=how,
                    filter_branch=filter_branch,
                    interpretation_executor=interpretation_executor,
                )
                arrays = library.global_index(arrays, global_start)
            except uproot.exceptions.KeyInFileError:
                if allow_missing:
                    continue
                else:
                    raise

            all_arrays.append(arrays)
            global_start = global_stop

    return library.concatenate(all_arrays)


class HasFields(Mapping):
    """
    Abstract class of behaviors for anything that "has fields," namely
    :doc:`uproot.models.RNTuple.RNTuple` and
    :doc:`uproot.models.RNTuple.RField`, which mostly consist of array-reading
    methods.

    A :doc:`uproot.behaviors.RNTuple.HasFields` is a Python ``Mapping``, which
    uses square bracket syntax to extract subfields:

    .. code-block:: python

        my_rntuple["field"]
        my_rntuple["field"]["subfield"]
        my_rntuple["field.subfield"]
        my_rntuple["field.subfield.subsubfield"]
        my_rntuple["field/subfield/subsubfield"]
        my_rntuple["field\\subfield\\subsubfield"]
    """

    @property
    def ntuple(self):
        """
        The :doc:`uproot.models.RNTuple.RNTuple` that this
        :doc:`uproot.models.RNTuple.HasFields` is part of.
        """
        return self._ntuple

    @property
    def num_entries(self):
        """
        The number of entries in the ``RNTuple``.
        """
        if isinstance(self, uproot.behaviors.RNTuple.RNTuple):
            if self._num_entries is None:
                self._num_entries = sum(x.num_entries for x in self.cluster_summaries)
            return self._num_entries
        return self.ntuple.num_entries

    @property
    def fields(self):
        """
        The list of :doc:`uproot.models.RNTuple.RField` directly under
        this :doc:`uproot.models.RNTuple.RNTuple` or
        :doc:`uproot.models.RNTuple.RField` (i.e. not recursive).
        """
        if self._fields is None:
            rntuple = self.ntuple
            if isinstance(self, uproot.behaviors.RNTuple.RNTuple):
                fields = [
                    rntuple.all_fields[i]
                    for i, f in enumerate(rntuple.field_records)
                    if f.parent_field_id == i
                ]
            else:
                fields = [
                    rntuple.all_fields[i]
                    for i, f in enumerate(rntuple.field_records)
                    if f.parent_field_id == self._fid
                    and f.parent_field_id != i
                    and not rntuple.all_fields[i].is_ignored
                ]
                # If the child field is anonymous, we return the grandchildren
                if len(fields) == 1 and fields[0].is_anonymous:
                    fields = fields[0].fields
            self._fields = fields
        return self._fields

    @property
    def path(self):
        """
        The full path of the field in the :doc:`uproot.models.RNTuple.RNTuple`. When it is
        the ``RNTuple`` itself, this is ``"."``.

        Note that this is not the full path within the ROOT file.
        """
        if isinstance(self, uproot.behaviors.RNTuple.RNTuple):
            return "."
        # For some anonymous fields, the path is not available
        if self.is_anonymous or self.is_ignored:
            return None
        if self._path is None:
            path = self.name
            parent = self.parent
            field = self
            while not isinstance(parent, uproot.behaviors.RNTuple.RNTuple):
                if not parent.is_anonymous:
                    path = f"{parent.name}.{path}"
                field = parent
                parent = field.parent
            self._path = path
        return self._path

    def to_akform(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        ak_add_doc=False,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            ak_add_doc (bool | dict ): If True and ``library="ak"``, add the RField ``description``
                to the Awkward ``__doc__`` parameter of the array.
                if dict = {key:value} and ``library="ak"``, add the RField ``value`` to the
                Awkward ``key`` parameter of the array.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns a 2-tuple where the first entry is the Awkward Form with the structure of the data in the ``RNTuple`` or ``RField``,
        and the second entry is the relative path of the requested RField. The second entry is needed in cases where the requested RField
        is a subfield of a collection, which requires constructing the form with information about the parent field.
        """
        ak = uproot.extras.awkward()

        keys = self.keys(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            filter_branch=filter_branch,
        )
        rntuple = self.ntuple

        top_names = []
        record_list = []
        if self is rntuple:
            field_path = None
            for field in self.fields:
                # the field needs to be in the keys or be a parent of a field in the keys
                if any(
                    key.startswith(f"{field.name}.") or key == field.name
                    for key in keys
                ):
                    top_names.append(field.name)
                    record_list.append(
                        rntuple.field_form(field.field_id, keys, ak_add_doc=ak_add_doc)
                    )
        else:
            # If it is a subfield of a collection, we need to include the collection in the keys
            path_keys = self.path.split(".")
            top_collection = None
            tmp_field = self.ntuple
            field_path = [self.name]
            for i, key in enumerate(path_keys):
                tmp_field = tmp_field[key]
                if (
                    tmp_field.record.struct_role
                    == uproot.const.RNTupleFieldRole.COLLECTION
                ):
                    top_collection = tmp_field
                    field_path = path_keys[i:]
                    break
            # Always use the full path for keys
            # Also include the field itself
            keys = [self.path] + [f"{self.path}.{k}" for k in keys]
            if top_collection is None:
                # The field needs to be in the keys or be a parent of a field in the keys
                if any(
                    key.startswith(f"{self.path}.") or key == self.path for key in keys
                ):
                    top_names.append(self.name)
                    record_list.append(
                        rntuple.field_form(self.field_id, keys, ak_add_doc=ak_add_doc)
                    )
            else:
                keys += [top_collection.path]
                top_names.append(top_collection.name)
                record_list.append(
                    rntuple.field_form(
                        top_collection.field_id, keys, ak_add_doc=ak_add_doc
                    )
                )

        parameters = None
        if isinstance(ak_add_doc, bool) and ak_add_doc and self.description != "":
            parameters = {"__doc__": self.description}
        elif isinstance(ak_add_doc, dict) and self is not rntuple:
            parameters = {
                key: self.__getattribute__(value) for key, value in ak_add_doc.items()
            }

        form = ak.forms.RecordForm(
            record_list, top_names, form_key="toplevel", parameters=parameters
        )
        return (form, field_path)

    def arrays(
        self,
        expressions=None,  # TODO: Not implemented yet
        cut=None,  # TODO: Not implemented yet
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        aliases=None,  # TODO: Not implemented yet
        language=uproot.language.python.python_language,  # TODO: Not implemented yet
        entry_start=None,
        entry_stop=None,
        decompression_executor=None,  # TODO: Not implemented yet
        array_cache="inherit",
        library="ak",  # TODO: Not implemented yet
        backend="cpu",
        interpreter="cpu",
        ak_add_doc=False,
        how=None,
        virtual=False,
        access_log=None,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        interpretation_executor=None,
        filter_branch=unset,
    ):
        """
        Args:
            expressions (None, str, or list of str): Names of ``RFields`` or
                aliases to convert to arrays or mathematical expressions of them.
                Uses the ``language`` to evaluate. If None, all ``RFields``
                selected by the filters are included. (Not implemented yet.)
            cut (None or str): If not None, this expression filters all of the
                ``expressions``. (Not implemented yet.)
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool, or None): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. If the function
                returns False or None, the ``RField`` is excluded; if the function
                returns True, it is included.
            aliases (None or dict of str \u2192 str): Mathematical expressions that
                can be used in ``expressions`` or other aliases.
                Uses the ``language`` engine to evaluate. (Not implemented yet.)
            language (:doc:`uproot.language.Language`): Language used to interpret
                the ``expressions`` and ``aliases``. (Not implemented yet.)
            entry_start (None or int): The first entry to include. If None, start
                at zero. If negative, count from the end, like a Python slice.
            entry_stop (None or int): The first entry to exclude (i.e. one greater
                than the last entry to include). If None, stop at
                :ref:`uproot.behaviors.RNTuple.RNTuple.num_entries`. If negative,
                count from the end, like a Python slice.
            decompression_executor (None or Executor with a ``submit`` method): The
                executor that is used to decompress ``RPages``; if None, the
                file's :ref:`uproot.reading.ReadOnlyFile.decompression_executor`
                is used. (Not implemented yet.)
            array_cache ("inherit", None, MutableMapping, or memory size): Cache of arrays;
                if "inherit", use the file's cache; if None, do not use a cache;
                if a memory size, create a new cache of this size.
            library (str or :doc:`uproot.interpretation.library.Library`): The library
                that is used to represent arrays. Options are ``"np"`` for NumPy,
                ``"ak"`` for Awkward Array, and ``"pd"`` for Pandas. (Not implemented yet.)
            backend (str): The backend Awkward Array will use.
            interpreter (str): If "cpu" will use cpu to interpret raw data. If "gpu" and
                ``backend="cuda"`` will use KvikIO bindings to CuFile and nvCOMP to
                interpret raw data on gpu if available.
            ak_add_doc (bool | dict ): If True and ``library="ak"``, add the RField ``description``
                to the Awkward ``__doc__`` parameter of the array.
                if dict = {key:value} and ``library="ak"``, add the RField ``value`` to the
                Awkward ``key`` parameter of the array.
            how (None, str, or container type): Library-dependent instructions
                for grouping. The only recognized container types are ``tuple``,
                ``list``, and ``dict``. Note that the container *type itself*
                must be passed as ``how``, not an instance of that type (i.e.
                ``how=tuple``, not ``how=()``).
            virtual (bool): If True, return virtual Awkward arrays, meaning that the data will not be
                loaded into memory until it is accessed.
            access_log (None or object with a ``__iadd__`` method): If an access_log is
                provided, e.g. a list, all materializations of the arrays are
                tracked inside this reference. Only applies if ``virtual=True``.
            interpretation_executor (None): This argument is not used and is only included for now
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns a group of arrays from the ``RNTuple``.

        For example:

        .. code-block:: python

            >>> my_ntuple.arrays()
            <Array [{my_vector: [1, 2]}, {...}] type='2 * {my_vector: var * int64}'>

        See also :ref:`uproot.behaviors.RNTuple.HasFields.array` to read a single
        ``RField`` as an array.

        See also :ref:`uproot.behaviors.RNTuple.HasFields.iterate` to iterate over
        the array in contiguous ranges of entries.
        """

        # This temporarily provides basic functionality while expressions are properly implemented
        if expressions is not None:
            if filter_name == no_filter:
                filter_name = expressions
            else:
                raise ValueError(
                    "Expressions are not supported yet. They are currently equivalent to filter_name."
                )

        if virtual:
            # some kwargs can't be used with virtual arrays
            err = "'{}' cannot be used with 'virtual=True'".format
            if how is not None:
                raise ValueError(err("how"))
            if library != "ak":
                raise ValueError(err("library"))
            if expressions is not None:
                raise ValueError(err("expressions"))
            if cut is not None:
                raise ValueError(err("cut"))
            if aliases is not None:
                raise ValueError(err("aliases"))
        else:
            # some kwargs can't be used with eager arrays
            err = "'{}' cannot be used with 'virtual=False'".format
            if access_log is not None:
                raise ValueError(err("access_log"))

        entry_start, entry_stop = (
            uproot.behaviors.TBranch._regularize_entries_start_stop(
                self.num_entries, entry_start, entry_stop
            )
        )
        library = uproot.interpretation.library._regularize_library(library)

        clusters = self.ntuple.cluster_summaries
        cluster_starts = numpy.array([c.num_first_entry for c in clusters])
        start_cluster_idx = (
            numpy.searchsorted(cluster_starts, entry_start, side="right") - 1
        )
        stop_cluster_idx = numpy.searchsorted(cluster_starts, entry_stop, side="right")
        cluster_num_entries = numpy.sum(
            [c.num_entries for c in clusters[start_cluster_idx:stop_cluster_idx]],
            dtype=int,
        )

        array_cache = _regularize_array_cache(array_cache, self.ntuple._file)

        form, field_path = self.to_akform(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            filter_branch=filter_branch,
            ak_add_doc=ak_add_doc,
        )

        # only read columns mentioned in the awkward form
        target_cols = []
        container_dict = {}
        _recursive_find(form, target_cols)

        if interpreter == "gpu" and backend == "cuda":
            clusters_datas = self.ntuple.gpu_read_clusters(
                target_cols, start_cluster_idx, stop_cluster_idx
            )
            clusters_datas._decompress()
            content_dict = self.ntuple.gpu_deserialize_decompressed_content(
                clusters_datas,
                start_cluster_idx,
                stop_cluster_idx,
            )

        for key in target_cols:
            if "column" in key:
                key_nr = int(key.split("-")[1])
                # Find how many elements should be padded at the beginning
                n_padding = self.ntuple.column_records[key_nr].first_element_index
                n_padding -= (
                    cluster_starts[start_cluster_idx] if start_cluster_idx >= 0 else 0
                )
                n_padding = max(n_padding, 0)
                dtype = None
                if interpreter == "cpu":
                    content_generator = partial(
                        self.ntuple.read_cluster_range,
                        key_nr,
                        start_cluster_idx,
                        stop_cluster_idx,
                        missing_element_padding=n_padding,
                        array_cache=array_cache,
                        access_log=access_log,
                    )
                    if virtual:
                        total_length, _, dtype = (
                            self.ntuple._expected_array_length_starts_dtype(
                                key_nr,
                                start_cluster_idx,
                                stop_cluster_idx,
                                missing_element_padding=n_padding,
                            )
                        )
                        if "cardinality" in key:
                            total_length -= 1
                        content = (total_length, content_generator)
                    else:
                        content = content_generator()
                elif interpreter == "gpu" and backend == "cuda":
                    content = content_dict[key_nr]
                elif interpreter == "gpu":
                    raise NotImplementedError(
                        f"Backend {backend} GDS support not implemented."
                    )
                else:
                    raise NotImplementedError(f"Backend {backend} not implemented.")
                dtype_byte = self.ntuple.column_records[key_nr].type
                _fill_container_dict(container_dict, content, key, dtype_byte, dtype)

        cluster_offset = (
            cluster_starts[start_cluster_idx] if start_cluster_idx >= 0 else 0
        )
        entry_start -= cluster_offset
        entry_stop -= cluster_offset
        arrays = uproot.extras.awkward().from_buffers(
            form,
            cluster_num_entries,
            container_dict,
            backend="cuda" if interpreter == "gpu" and backend == "cuda" else "cpu",
        )[entry_start:entry_stop]

        arrays = uproot.extras.awkward().to_backend(arrays, backend=backend)
        # no longer needed; save memory
        del container_dict

        # If we constructed some parent fields, we need to get back to the requested field
        if field_path is not None:
            for field in field_path[:-1]:
                if field in arrays.fields:
                    arrays = arrays[field]
                # tuples are a trickier since indices no longer match
                else:
                    if field.isdigit() and arrays.fields == ["0"]:
                        arrays = arrays["0"]
                    else:
                        raise AssertionError(
                            "The array was not constructed correctly. Please report this issue."
                        )

        # FIXME: This is not right, but it might temporarily work
        if library.name == "np":
            return arrays.to_numpy()

        # TODO: This should be done with library.group, if possible
        if how is tuple:
            arrays = tuple(arrays[f] for f in arrays.fields)
        elif how is list:
            arrays = [arrays[f] for f in arrays.fields]
        elif how is dict:
            arrays = {f: arrays[f] for f in arrays.fields}
        elif how is not None:
            raise ValueError(
                f"unrecognized 'how' parameter: {how}. Options are None, tuple, list and dict."
            )
        return arrays

    def __array__(self, *args, **kwargs):
        if isinstance(self, uproot.behaviors.RNTuple.RNTuple):
            out = self.arrays(library="np")
        else:
            out = self.array(library="np")
        if args == () and kwargs == {}:
            return out
        else:
            return numpy.array(out, *args, **kwargs)

    def iterate(
        self,
        expressions=None,  # TODO: Not implemented yet
        cut=None,  # TODO: Not implemented yet
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        aliases=None,  # TODO: Not implemented yet
        language=uproot.language.python.python_language,  # TODO: Not implemented yet
        entry_start=None,
        entry_stop=None,
        step_size="100 MB",
        decompression_executor=None,  # TODO: Not implemented yet
        library="ak",  # TODO: Not implemented yet
        backend="cpu",
        interpreter="cpu",
        ak_add_doc=False,
        how=None,
        report=False,  # TODO: Not implemented yet
        # For compatibility reasons we also accepts kwargs meant for TTrees
        interpretation_executor=None,
        filter_branch=unset,
    ):
        """
        Args:
            expressions (None, str, or list of str): Names of ``RFields`` or
                aliases to convert to arrays or mathematical expressions of them.
                Uses the ``language`` to evaluate. If None, all ``RFields``
                selected by the filters are included. (Not implemented yet.)
            cut (None or str): If not None, this expression filters all of the
                ``expressions``. (Not implemented yet.)
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``EFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool, or None): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. If the function
                returns False or None, the ``RField`` is excluded; if the function
                returns True, it is included.
            aliases (None or dict of str \u2192 str): Mathematical expressions that
                can be used in ``expressions`` or other aliases.
                Uses the ``language`` engine to evaluate. (Not implemented yet.)
            language (:doc:`uproot.language.Language`): Language used to interpret
                the ``expressions`` and ``aliases``. (Not implemented yet.)
            entry_start (None or int): The first entry to include. If None, start
                at zero. If negative, count from the end, like a Python slice.
            entry_stop (None or int): The first entry to exclude (i.e. one greater
                than the last entry to include). If None, stop at
                :ref:`uproot.behaviors.RNTuple.HasFields.num_entries`. If negative,
                count from the end, like a Python slice.
            step_size (int or str): If an integer, the maximum number of entries to
                include in each iteration step; if a string, the maximum memory size
                to include. The string must be a number followed by a memory unit,
                such as "100 MB".
            decompression_executor (None or Executor with a ``submit`` method): The
                executor that is used to decompress ``RPages``; if None, the
                file's :ref:`uproot.reading.ReadOnlyFile.decompression_executor`
                is used. (Not implemented yet.)
            library (str or :doc:`uproot.interpretation.library.Library`): The library
                that is used to represent arrays. Options are ``"np"`` for NumPy,
                ``"ak"`` for Awkward Array, and ``"pd"`` for Pandas. (Not implemented yet.)
            ak_add_doc (bool | dict ): If True and ``library="ak"``, add the RField ``description``
                to the Awkward ``__doc__`` parameter of the array.
                if dict = {key:value} and ``library="ak"``, add the RField ``value`` to the
                Awkward ``key`` parameter of the array. (Not implemented yet.)
            how (None, str, or container type): Library-dependent instructions
                for grouping. The only recognized container types are ``tuple``,
                ``list``, and ``dict``. Note that the container *type itself*
                must be passed as ``how``, not an instance of that type (i.e.
                ``how=tuple``, not ``how=()``).
            report (bool): If True, this generator yields
                (arrays, :doc:`uproot.behaviors.TBranch.Report`) pairs; if False,
                it only yields arrays. The report has data about the ``TFile``,
                ``RNTuple``, and global and local entry ranges. (Not implemented yet.)
            interpretation_executor (None): This argument is not used and is only included for now
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Iterates through contiguous chunks of entries from the ``RNTuple``.

        For example:

        .. code-block:: python

            >>> for array in rntuple.iterate(filter_name=["x", "y"], step_size=100):
            ...     # each of the following have 100 entries
            ...     array["x"], array["y"]

        See also :ref:`uproot.behaviors.RNTuple.HasFields.arrays` to read
        everything in a single step, without iteration.

        See also :doc:`uproot.behaviors.RNTuple.iterate` to iterate over many
        files.
        """
        # This temporarily provides basic functionality while expressions are properly implemented
        if expressions is not None:
            if filter_name == no_filter:
                filter_name = expressions
            else:
                raise ValueError(
                    "Expressions are not supported yet. They are currently equivalent to filter_name."
                )

        entry_start, entry_stop = (
            uproot.behaviors.TBranch._regularize_entries_start_stop(
                self.ntuple.num_entries, entry_start, entry_stop
            )
        )

        akform, _ = self.to_akform(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            filter_branch=filter_branch,
            ak_add_doc=ak_add_doc,
        )

        step_size = _regularize_step_size(
            self.ntuple, akform, step_size, entry_start, entry_stop
        )
        # TODO: This can be done more efficiently
        for start in range(0, self.num_entries, step_size):
            yield self.arrays(
                filter_name=filter_name,
                filter_typename=filter_typename,
                filter_field=filter_field,
                entry_start=start,
                entry_stop=start + step_size,
                library=library,
                backend=backend,
                interpreter=interpreter,
                how=how,
                filter_branch=filter_branch,
            )

    def keys(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        ignore_duplicates=False,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            ignore_duplicates (bool): If True, return a set of the keys; otherwise, return the full list of keys.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns the names of the subfields as a list of strings.
        """
        return list(
            self.iterkeys(
                filter_name=filter_name,
                filter_typename=filter_typename,
                filter_field=filter_field,
                recursive=recursive,
                full_paths=full_paths,
                ignore_duplicates=ignore_duplicates,
                filter_branch=filter_branch,
            )
        )

    def values(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns the subfields as a list of
        :doc:`uproot.behaviors.RField.RField`.

        (Note: with ``recursive=False``, this is the same as
        :ref:`uproot.behaviors.TBranch.HasFields.fields`.)
        """
        return list(
            self.itervalues(
                filter_name=filter_name,
                filter_typename=filter_typename,
                filter_field=filter_field,
                recursive=recursive,
                filter_branch=filter_branch,
            )
        )

    def items(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns (name, field) pairs of the subfields as a list of 2-tuples
        of (str, :doc:`uproot.behaviors.RField.RField`).
        """
        return list(
            self.iteritems(
                filter_name=filter_name,
                filter_typename=filter_typename,
                filter_field=filter_field,
                recursive=recursive,
                full_paths=full_paths,
                filter_branch=filter_branch,
            )
        )

    def typenames(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns (name, typename) pairs of the subfields as a dict of
        str \u2192 str.
        """
        return dict(
            self.itertypenames(
                filter_name=filter_name,
                filter_typename=filter_typename,
                filter_field=filter_field,
                recursive=recursive,
                full_paths=full_paths,
                filter_branch=filter_branch,
            )
        )

    def iterkeys(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        ignore_duplicates=False,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            ignore_duplicates (bool): If True, return a set of the keys; otherwise, return the full list of keys.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.


        Returns the names of the subfields as an iterator over strings.
        """
        for k, _ in self.iteritems(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            recursive=recursive,
            full_paths=full_paths,
            ignore_duplicates=ignore_duplicates,
            filter_branch=filter_branch,
        ):
            yield k

    def itervalues(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns the subfields as an iterator over
        :doc:`uproot.behaviors.RField.RField`.

        (Note: with ``recursive=False``, this is the same as
        :ref:`uproot.behaviors.RField.HasFields.fields`.)
        """
        for _, v in self.iteritems(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            recursive=recursive,
            full_paths=False,
            filter_branch=filter_branch,
        ):
            yield v

    def iteritems(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        ignore_duplicates=False,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            ignore_duplicates (bool): If True, return a set of the keys; otherwise, return the full list of keys.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.


        Returns (name, field) pairs of the subfields as an iterator over
        2-tuples of (str, :doc:`uproot.behaviors.RField.RField`).
        """
        if filter_branch is not unset:
            warnings.warn(
                "the filter_branch kwarg should not be used for RNTuples",
                DeprecationWarning,
                stacklevel=1,
            )
            filter_field = filter_branch

        filter_name = uproot._util.regularize_filter(filter_name)
        filter_typename = uproot._util.regularize_filter(filter_typename)
        if filter_field is None:
            filter_field = no_filter
        elif callable(filter_field):
            pass
        else:
            raise TypeError(
                f"filter_field must be None or a function: RField -> bool, not {filter_field!r}"
            )

        keys_set = set()

        for field in self.fields:
            if (
                (
                    filter_name is no_filter
                    or _filter_name_deep(filter_name, self, field)
                )
                and (filter_typename is no_filter or filter_typename(field.typename))
                and (filter_field is no_filter or filter_field(field))
            ):
                if field.is_anonymous or (ignore_duplicates and field.name in keys_set):
                    pass
                else:
                    keys_set.add(field.name)
                    yield field.name, field

            if recursive:
                for k1, v in field.iteritems(
                    recursive=recursive,
                    filter_name=no_filter,
                    filter_typename=filter_typename,
                    filter_field=filter_field,
                    full_paths=full_paths,
                ):
                    k2 = (
                        f"{field.name}.{k1}"
                        if full_paths and not field.is_anonymous
                        else k1
                    )
                    if filter_name is no_filter or _filter_name_deep(
                        filter_name, self, v
                    ):
                        if ignore_duplicates and k2 in keys_set:
                            pass
                        else:
                            keys_set.add(k2)
                            yield k2, v

    def itertypenames(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        full_paths=True,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields``s by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, descend into any nested subfields.
                If False, only return the names of the top fields.
            full_paths (bool): If True, include the full path to each subfield
                with periods (``.``); otherwise, use the descendant's name as
                the output name.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns (name, typename) pairs of the subfields as an iterator over
        2-tuples of (str, str).
        """
        for k, v in self.iteritems(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            recursive=recursive,
            full_paths=full_paths,
            filter_branch=filter_branch,
        ):
            yield k, v.typename

    def _ipython_key_completions_(self):
        """
        Supports key-completion in an IPython or Jupyter kernel.
        """
        return self.iterkeys()

    def num_entries_for(
        self,
        memory_size,
        expressions=None,  # TODO: Not implemented yet
        cut=None,  # TODO: Not implemented yet
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        aliases=None,  # TODO: Not implemented yet
        language=uproot.language.python.python_language,  # TODO: Not implemented yet
        entry_start=None,
        entry_stop=None,
        # For compatibility reasons we also accepts kwargs meant for TTrees
        filter_branch=unset,
    ):
        """
        Args:
            memory_size (int or str): An integer is interpreted as a number of
                bytes and a string must be a number followed by a unit, such as
                "100 MB".
            expressions (None, str, or list of str): Names of ``RFields`` or
                aliases to convert to arrays or mathematical expressions of them.
                Uses the ``language`` to evaluate. If None, all ``RFields``
                selected by the filters are included. (Not implemented yet.)
            cut (None or str): If not None, this expression filters all of the
                ``expressions``. (Not implemented yet.)
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RField.RField` \u2192 bool, or None): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RField.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            aliases (None or dict of str \u2192 str): Mathematical expressions that
                can be used in ``expressions`` or other aliases.
                Uses the ``language`` engine to evaluate.
            language (:doc:`uproot.language.Language`): Language used to interpret
                the ``expressions`` and ``aliases``.
            entry_start (None or int): The first entry to include. If None, start
                at zero. If negative, count from the end, like a Python slice.
            entry_stop (None or int): The first entry to exclude (i.e. one greater
                than the last entry to include). If None, stop at
                :ref:`uproot.behaviors.RNTuple.HasFields.num_entries`. If negative,
                count from the end, like a Python slice.
            filter_branch (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool): An alias for ``filter_field`` included
                for compatibility with software that was used for :doc:`uproot.behaviors.TBranch.TBranch`. This argument should not be used
                and will be removed in a future version.

        Returns an *approximate* step size as a number of entries to read
        a given ``memory_size`` in each step.

        This method does not actually read the ``RField`` data or compute any
        expressions to arrive at its estimate. It only uses metadata from the
        already-loaded ``RNTuple``; it only needs ``language`` to parse the
        expressions, not to evaluate them.

        This is the algorithm that
        :ref:`uproot.behaviors.RNTuple.HasFields.iterate` uses to convert a
        ``step_size`` expressed in memory units into a number of entries.
        """
        # This temporarily provides basic functionality while expressions are properly implemented
        if expressions is not None:
            if filter_name == no_filter:
                filter_name = expressions
            else:
                raise ValueError(
                    "Expressions are not supported yet. They are currently equivalent to filter_name."
                )

        target_num_bytes = uproot._util.memory_size(memory_size)

        entry_start, entry_stop = (
            uproot.behaviors.TBranch._regularize_entries_start_stop(
                self.ntuple.num_entries, entry_start, entry_stop
            )
        )

        akform, _ = self.to_akform(
            filter_name=filter_name,
            filter_typename=filter_typename,
            filter_field=filter_field,
            filter_branch=filter_branch,
        )

        if len(akform.contents) == 0:
            return

        return _num_entries_for(self, akform, target_num_bytes, entry_start, entry_stop)

    def __getitem__(self, where):
        original_where = where

        if uproot._util.isint(where):
            return self.fields[where]
        elif isinstance(where, str):
            where = uproot._util.ensure_str(where)
            where = where.replace("/", ".").replace("\\", ".")
        else:
            raise TypeError(f"where must be an integer or a string, not {where!r}")

        if where.startswith("."):
            recursive = False
            where = where[1:]
        else:
            recursive = True

        if self._lookup is None:
            self._lookup = {f.name: f for f in self.fields}
        got = self._lookup.get(where)
        if got is not None:
            return got

        if "." in where:
            this = self
            try:
                for piece in where.split("."):
                    if piece != "":
                        this = this[piece]
            except uproot.KeyInFileError:
                raise uproot.KeyInFileError(
                    original_where,
                    keys=self.keys(recursive=recursive),
                    file_path=self.ntuple.parent._file.file_path,
                    object_path=self.path,
                ) from None
            return this

        elif recursive:
            got = _get_recursive(self, where)
            if got is not None:
                return got
            else:
                raise uproot.KeyInFileError(
                    original_where,
                    keys=self.keys(recursive=recursive),
                    file_path=self.ntuple.parent._file.file_path,
                    object_path=self.path,
                )

        else:
            raise uproot.KeyInFileError(
                original_where,
                keys=self.keys(recursive=recursive),
                file_path=self._file.file_path,
                object_path=self.object_path,
            )

    def __iter__(self):
        yield from self.fields

    def __len__(self):
        return len(self.fields)

    def show(
        self,
        *,
        filter_name=no_filter,
        filter_typename=no_filter,
        filter_field=no_filter,
        recursive=True,
        max_width=80,
        stream=sys.stdout,
        **kwargs,
    ):
        """
        Args:
            filter_name (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by name.
            filter_typename (None, glob string, regex string in ``"/pattern/i"`` syntax, function of str \u2192 bool, or iterable of the above): A
                filter to select ``RFields`` by type.
            filter_field (None or function of :doc:`uproot.models.RNTuple.RField` \u2192 bool, or None): A
                filter to select ``RFields`` using the full
                :doc:`uproot.models.RNTuple.RField` object. The ``RField`` is
                included if the function returns True, excluded if it returns False.
            recursive (bool): If True, recursively descend into subfields.
            max_width (int): Maximum number of characters to display in a line.
            stream (object with a ``write(str)`` method): Stream to write the
                output to.

        Interactively display the ``RFields``.

        For example,

        .. code-block::

            >>> my_ntuple.show()
            my_ntuple (ROOT::RNTuple)
            Description: The description of the ntuple
            ├─ my_int (std::int64_t)
            │  Description: The description of the field
            ├─ jagged_list (std::vector<std::int64_t>)
            ├─ nested_list (std::vector<std::vector<std::int64_t>>)
            ├─ struct (MyStruct)
            │  ├─ x (std::int64_t)
            │  └─ y (std::int64_t)
            └─ other_struct (OtherStruct)
                ├─ a (SubStruct)
                │  Description: The description of the subfield
                │  ├─ x (std::int64_t)
                │  └─ y (std::int64_t)
                └─ b (std::int64_t)
        """
        elbow = "└─ "
        pipe = "│  "
        tee = "├─ "
        blank = "   "

        def recursive_show(field, header="", first=True, last=True, recursive=True):
            outstr = f"""{header}{"" if first else (elbow if last else tee)}{field.name} ({'ROOT::RNTuple' if isinstance(field, uproot.behaviors.RNTuple.RNTuple) else field.typename})"""
            stream.write(outstr[:max_width] + "\n")
            if field.description != "":
                outstr = f"""{header}{'' if first else (blank if last else pipe)}Description: {field.description}"""
                stream.write(outstr[:max_width] + "\n")
            if len(field) > 0 and (recursive or first):
                subfields = list(
                    field.itervalues(
                        filter_name=filter_name,
                        filter_typename=filter_typename,
                        filter_field=filter_field,
                        recursive=False,
                    )
                )
                for i, subfield in enumerate(subfields):
                    recursive_show(
                        subfield,
                        header=f"{header}{'' if first else (blank if last else pipe)}",
                        first=False,
                        last=i == len(subfields) - 1,
                        recursive=recursive,
                    )

        recursive_show(self, recursive=recursive)

    @property
    def source(self) -> uproot.source.chunk.Source | None:
        """Returns the associated source of data for this container, if it exists

        Returns: uproot.source.chunk.Source or None
        """
        if isinstance(self.ntuple._file, uproot.reading.ReadOnlyFile):
            return self.ntuple._file.source
        return None


class RNTuple(HasFields):
    """
    Behaviors for an ``RNTuple``, which mostly consist of array-reading methods.

    Since a :doc:`uproot.behaviors.RNTuple.RNTuple` is a
    :doc:`uproot.behaviors.RNTuple.HasFields`, it is also a Python
    ``Mapping``, which uses square bracket syntax to extract subbranches:

    .. code-block:: python

        my_rntuple["field"]
        my_rntuple["field"]["subfield"]
        my_rntuple["field.subfield"]
        my_rntuple["field.subfield.subsubfield"]
        my_rntuple["field/subfield/subsubfield"]
        my_rntuple["field\\subfield\\subsubfield"]
    """

    def __repr__(self):
        if len(self) == 0:
            return f"<{self.classname} {self.name!r} at 0x{id(self):012x}>"
        else:
            return f"<{self.classname} {self.name!r} ({len(self)} fields) at 0x{id(self):012x}>"

    @property
    def name(self):
        """
        Name of the ``RNTuple``.
        """
        return self.header.ntuple_name

    @property
    def description(self):
        """
        Description of the ``RNTuple``.
        """
        return self.header.ntuple_description

    @property
    def object_path(self):
        """
        Object path of the ``RNTuple``.
        """
        return self.parent.object_path

    @property
    def cache_key(self):
        """
        String that uniquely specifies this ``RNTuple`` in its path, to use as
        part of object and array cache keys.
        """
        return f"{self.parent.cache_key}{self.name};{self.parent.fCycle}"


def _filter_name_deep(filter_name, hasfields, field):
    shallow = name = field.name
    if filter_name(name):
        return True
    while field is not hasfields:
        field = field.parent
        if field is not hasfields:
            name = field.name + "." + name
    if name != shallow and filter_name(name):
        return True
    return filter_name("." + name)


def _get_recursive(hasfields, where):
    if hasfields._lookup is None:
        hasfields._lookup = {f.name: f for f in hasfields.fields}
    got = hasfields._lookup.get(where)
    if got is not None:
        return got
    for field in hasfields.fields:
        got = _get_recursive(field, where)
        if got is not None:
            return got
    else:
        return None


def _num_entries_for(ntuple, akform, target_num_bytes, entry_start, entry_stop):
    # TODO: there might be a better way to estimate the number of entries
    clusters = ntuple.cluster_summaries
    cluster_starts = numpy.array([c.num_first_entry for c in clusters])

    start_cluster_idx = (
        numpy.searchsorted(cluster_starts, entry_start, side="right") - 1
    )
    stop_cluster_idx = numpy.searchsorted(cluster_starts, entry_stop, side="right")

    target_cols = []
    _recursive_find(akform, target_cols)

    total_bytes = 0
    for key in target_cols:
        if "column" in key:
            key_nr = int(key.split("-")[1])
            for cluster in range(start_cluster_idx, stop_cluster_idx):
                pages = ntuple.page_link_list[cluster][key_nr].pages
                total_bytes += sum(page.locator.num_bytes for page in pages)

    total_entries = entry_stop - entry_start
    if total_bytes == 0:
        num_entries = 0
    else:
        num_entries = round(target_num_bytes * total_entries / total_bytes)
    if num_entries <= 0:
        return 1
    else:
        return num_entries


def _regularize_step_size(ntuple, akform, step_size, entry_start, entry_stop):
    if uproot._util.isint(step_size):
        return step_size
    target_num_bytes = uproot._util.memory_size(
        step_size,
        "number of entries or memory size string with units "
        f"(such as '100 MB') required, not {step_size!r}",
    )
    return _num_entries_for(ntuple, akform, target_num_bytes, entry_start, entry_stop)


def _recursive_find(form, res):
    ak = uproot.extras.awkward()

    if hasattr(form, "form_key") and form.form_key not in res:
        res.append(form.form_key)
    if hasattr(form, "contents"):
        for c in form.contents:
            _recursive_find(c, res)
    if hasattr(form, "content") and issubclass(type(form.content), ak.forms.Form):
        _recursive_find(form.content, res)


def _cupy_insert(arr, obj, value):
    # obj is assumed to be sorted
    # both arr and obj are assumed to be flat arrays
    cupy = uproot.extras.cupy()
    out_size = arr.size + obj.size
    out = cupy.empty(out_size, dtype=arr.dtype)
    src_i = 0
    dst_i = 0
    for idx in obj.get():
        n = idx - src_i
        if n > 0:
            out[dst_i : dst_i + n] = arr[src_i : src_i + n]
            dst_i += n
            src_i += n
        out[dst_i] = value
        dst_i += 1
    if src_i < arr.size:
        out[dst_i:] = arr[src_i:]
    return out


def _fill_container_dict(container_dict, content, key, dtype_byte, dtype):
    ak = uproot.extras.awkward()
    Numpy = ak._nplikes.numpy.Numpy

    if isinstance(content, tuple):
        # Virtual arrays not yet implemented for GPU
        array_library_string = "numpy"
        virtual = True
        length = int(content[0])
        raw_generator = content[1]
    else:
        virtual = False
        array_library_string = uproot._util.get_array_library(content)
        length = len(content)

        def raw_generator():
            return content

    if virtual:
        from packaging.version import Version

        if Version(ak.__version__) < Version("2.8.11"):
            raise ImportError("Virtual arrays require Awkward version 2.8.11 or later")
        VirtualNDArray = ak._nplikes.virtual.VirtualNDArray

    library = numpy if array_library_string == "numpy" else uproot.extras.cupy()

    if "cardinality" in key:

        def generator():
            materialized = raw_generator()
            materialized = library.diff(materialized)
            return materialized

        if virtual:
            virtual_array = VirtualNDArray(
                Numpy.instance(), shape=(length,), dtype=dtype, generator=generator
            )
            container_dict[f"{key}-data"] = virtual_array
        else:
            container_dict[f"{key}-data"] = generator()

    elif "optional" in key:

        def generator():
            # We need to convert from a ListOffsetArray to an IndexedOptionArray
            materialized = raw_generator()
            diff = library.diff(materialized)
            missing = library.nonzero(diff == 0)[0]
            missing -= library.arange(len(missing), dtype=missing.dtype)
            dtype = "int64" if materialized.dtype == library.int64 else "int32"
            indices = library.arange(len(materialized) - len(missing), dtype=dtype)
            if array_library_string == "numpy":
                indices = numpy.insert(indices, missing, -1)
            else:
                indices = _cupy_insert(indices, missing, -1)
            return indices[:-1]  # We need to delete the last index

        if virtual:
            virtual_array = VirtualNDArray(
                Numpy.instance(), shape=(length - 1,), dtype=dtype, generator=generator
            )
            container_dict[f"{key}-index"] = virtual_array
        else:
            container_dict[f"{key}-index"] = generator()

    elif dtype_byte == uproot.const.rntuple_col_type_to_num_dict["switch"]:

        def tag_generator():
            content = raw_generator()
            return content["tag"].astype(numpy.int8)

        def index_generator():
            content = raw_generator()
            tags = content["tag"].astype(numpy.int8)
            kindex = content["index"]
            # Find invalid variants and adjust buffers accordingly
            invalid = numpy.flatnonzero(tags == 0)
            kindex[invalid] = 0  # Might not be necessary, but safer
            return kindex

        def nones_index_generator():
            return library.array([-1], dtype=numpy.int64)

        if virtual:
            tag_virtual_array = VirtualNDArray(
                Numpy.instance(),
                shape=(length,),
                dtype=numpy.int8,
                generator=tag_generator,
            )
            container_dict[f"{key}-tags"] = tag_virtual_array
            index_virtual_array = VirtualNDArray(
                Numpy.instance(),
                shape=(length,),
                dtype=numpy.int64,
                generator=index_generator,
            )
            container_dict[f"{key}-index"] = index_virtual_array
            nones_index_virtual_array = VirtualNDArray(
                Numpy.instance(),
                shape=(1,),
                dtype=numpy.int64,
                generator=nones_index_generator,
            )
            container_dict["nones-index"] = nones_index_virtual_array
        else:
            container_dict[f"{key}-tags"] = tag_generator()
            container_dict[f"{key}-index"] = index_generator()
            container_dict["nones-index"] = nones_index_generator()
    else:
        if virtual:
            virtual_array = VirtualNDArray(
                Numpy.instance(), shape=(length,), dtype=dtype, generator=raw_generator
            )
            container_dict[f"{key}-data"] = virtual_array
            container_dict[f"{key}-offsets"] = virtual_array
        else:
            container_dict[f"{key}-data"] = content
            container_dict[f"{key}-offsets"] = content
